/*
 * Copyright 2000-2009
 * Wolfgang Denk, DENX Software Engineering, wd@denx.de.
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program isz distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <linux/types.h>

#include <command.h>
#include <common.h>
#include "../fs/ubifs/ubifs.h"
#include "ams_part.h"
#include "ams_ucmgr.h"
#include <version.h>

DECLARE_GLOBAL_DATA_PTR;
extern UC_MEM_Header_T  *uc_head;
static UC_MGR_Sys_Info_T sys_info;
static FS_HW_Info_T      hw_info;
AMS_FILEMAPPING_MGR_T *fmm=0;
static int ubi_loaded=FALSE;
static char cmd[255];
#define XMK_STR(x)  #x
#define MK_STR(x)   XMK_STR(x)
#define RETURN_ERROR(format,args...)  do{printf("%s:%d:"format"\r\n",__FUNCTION__,__LINE__ ,##args);return -1;}while(0)
#define is_digit(c)  ((c) >= '0' && (c) <= '9')
#define is_xdigit(c) (((c) >= '0' && (c) <= '9')    \
                      || ((c) >= 'a' && (c) <= 'f') \
                      || ((c) >= 'A' && (c) <= 'F'))
#define CMD_RET_USAGE cmd_usage(cmdtp)
static UI8_T _str[256]={0};
#define GET_STRING(str) (char*)memcpy(memset(_str,0,sizeof(_str)),(str),sizeof(str))
#define SET_STRING(str,value) (char*)memcpy(memset((str),0,sizeof(str)),(value),(strlen(value)<sizeof(str))?strlen(value):sizeof(str))

void init_system(void);
int  init_ubifs(void);
void set_mtdparts(void);
void recover_mfg(void);
int ubi_boot(void);
int jffs2_boot(void);
int acc_boot(void);
int acc_recovery(void);
BOOL_T verify_hwinfo(FS_HW_Info_T *hwinfo);
BOOL_T update_uc_by_hwinfo(void);
void verify_board_id(void);

int jffs2_load_filemapping(void);
static void Encrypt(char * data);
static int LA_get_password(char *passwd, uint timeout, uint max_size);
static int LA_check_uboot_passwd_in_env(void);
#ifdef CONFIG_USR_PRIVILEGED_MODE
static int LA_set_privi_mode(unsigned long mode);
static unsigned long LA_get_privi_mode(void);
static unsigned long LA_privileged_mode = FALSE;
#endif
static UC_MGR_Sys_Info_T default_ucinfo =
{
    CONFIG_MAGIC,                           /* magic                */
    0,                                      /*warm start entry*/
    {
        CONFIG_ETHADDR0,
        CONFIG_ETHADDR1,
        CONFIG_ETHADDR2,
        CONFIG_ETHADDR3,
        CONFIG_ETHADDR4,
        CONFIG_ETHADDR5
    },                                      /*eth addr*/
    MK_STR(CONFIG_SERIAL_NUM),              /* serial_no           */
    MK_STR(CONFIG_HARDWARE_VER),            /* hardware_ver   */
    MK_STR(CONFIG_MANUFACTURE_DATE),        /* manufacture_date */
    "",                                     /* Loader software version(len 11+1)            */
    "",                                     /* POST software version(len 11+1)              */
    "",                                     /* runtime version(len 11+1)                    */
    MK_STR(CONFIG_SERVICE_TAG),             /*server_tag              */
    MK_STR(CONFIG_MODEL_NUM),               /*model_num             */
    "",                                     /* service tag serial number (len 11+1)         */
    0,                                      /*sw_identifier*/
    CONFIG_PROJECT_ID,                      /* project_id         */
    CONFIG_BOARD_ID,                        /* board_id           */
    CONFIG_BAUDRATE,
    0,
    0,
    "",
    0,
    0,
    "",
    {""},
    0
};


int set_default_hwinfo(FS_HW_Info_T *hwinfo)
{
    if(hwinfo != NULL) {
        memset((unsigned char *)hwinfo, 0, sizeof(FS_HW_Info_T));
        hwinfo->board_id = CONFIG_BOARD_ID;
        hwinfo->project_id = CONFIG_PROJECT_ID;
        hwinfo->baudrate = CONFIG_BAUDRATE;
        hwinfo->mac_addr[0] = CONFIG_ETHADDR0;
        hwinfo->mac_addr[1] = CONFIG_ETHADDR1;
        hwinfo->mac_addr[2] = CONFIG_ETHADDR2;
        hwinfo->mac_addr[3] = CONFIG_ETHADDR3;
        hwinfo->mac_addr[4] = CONFIG_ETHADDR4;
        hwinfo->mac_addr[5] = CONFIG_ETHADDR5;
        strcpy((char *)hwinfo->agent_hw_ver, MK_STR(CONFIG_HARDWARE_VER));
        strcpy((char *)hwinfo->serial_no, MK_STR(CONFIG_SERIAL_NUM));
        strcpy((char *)hwinfo->manufacture_date, MK_STR(CONFIG_MANUFACTURE_DATE));
        strcpy((char *)hwinfo->service_tag, MK_STR(CONFIG_SERVICE_TAG));
        strcpy((char *)hwinfo->model_num_by_string, MK_STR(CONFIG_MODEL_NUM));
        hwinfo->special_boot_op = 0;
        hwinfo->capability = FS_HWINFO_CAPABILITY_CHANGE_LOADER_BACKDOOR_PASSWORD;
        hwinfo->check_sum = CheckSum((unsigned long)hwinfo, sizeof(FS_HW_Info_T) - sizeof(unsigned short));
        return RC_OK;
    }

    return RC_ERROR;
}

unsigned long ENV_COMMON_check_mac(unsigned char *value)
{
    int           i;
    unsigned char t;

    if (strlen((char *)value) != 17)
        goto error;

    for (i = 0; i < 17; )
    {

        t = value[i++];
        if (!is_xdigit(t))
            goto error;

        t = value[i++];
        if (!is_xdigit(t))
            goto error;

        if (i < 17)
        {
            t = value[i++];
            if (t != ':')
                goto error;
        }
    }

    t = (unsigned char)(simple_strtoul((char *)value, 0, 16) & 0xff);

    if (t & 0x01)
    {
error:
        return RC_ERROR;
    }

    return RC_OK;
}

/* ------------------------------------------------------------------------
 * FUNCTION NAME: HWINFO_read
 * PURPOSE: read the H/W information from flash.
 * INPUT:   None
 * OUTPUT:  None
 * RETURN:
 *          0(RC_OK)     = Success
 *          -1(RC_ERROR) = Fail
 * NOTES:
 */
int HWINFO_read(FS_HW_Info_T *hwinfo)
{
    int ret;
    unsigned short checksum;
    sprintf(cmd,"sf read %x %x %x",(unsigned int)hwinfo,CONFIG_HWINFO_PART_BASE_ADDR,sizeof(FS_HW_Info_T));

    if(0>(ret=run_command(cmd,0)))
        RETURN_ERROR("Failed to load HWINFO");
    checksum = (unsigned short)CheckSum((unsigned long)hwinfo, sizeof(FS_HW_Info_T) - sizeof(unsigned short));
    debug("%s, %d: [sizeof(FS_HW_Info_T)] %u\n", __FUNCTION__, __LINE__, sizeof(FS_HW_Info_T));
    debug("%s, %d: [checksum] 0x%08X, [hwinfo->check_sum] 0x%08X\n", __FUNCTION__, __LINE__, checksum, hwinfo->check_sum);
    if(checksum == 0xffff || checksum != hwinfo->check_sum)
        return RC_ERROR;

    return RC_OK;
}

/* ------------------------------------------------------------------------
 * FUNCTION NAME: HWINFO_write
 * PURPOSE: write the H/W information to flash.
 * INPUT:   None
 * OUTPUT:  None
 * RETURN:
 *          0(RC_OK)    = Success
 *          -1(RC_ERROR)= Fail
 * NOTES:
 */
int HWINFO_write(FS_HW_Info_T *hwinfo)
{
    int ret;
    /* Update checksum */
    hwinfo->check_sum =(unsigned short)CheckSum((unsigned long)hwinfo, sizeof(FS_HW_Info_T) - sizeof(unsigned short));

    /* write to hwinfo partition */
    sprintf(cmd,"sf erase %x %x",CONFIG_HWINFO_PART_BASE_ADDR,CONFIG_HWINFO_PART_SIZE);
    if(0>(ret=run_command(cmd,0)))
        RETURN_ERROR("Failed to sf erase %x %x",CONFIG_HWINFO_PART_BASE_ADDR,CONFIG_HWINFO_PART_SIZE);

    sprintf(cmd,"sf write %x %x %x",(int)hwinfo,CONFIG_HWINFO_PART_BASE_ADDR,sizeof(*hwinfo));
    if(0>(ret=run_command(cmd,0)))
        RETURN_ERROR("Failed to sf write %x %x",CONFIG_HWINFO_PART_BASE_ADDR,sizeof(*hwinfo));

    return RC_OK;
}
BOOL_T HWINFO_load(void)
{
    if(RC_OK!=HWINFO_read(&hw_info))
        return FALSE;
    return TRUE;
}

int scan_mac_from_string(unsigned char mac_addr[],char *mac_str )
{
    int i;
    char* end;
    for (i = 0; i < 6; i++) {
        mac_addr[i] = mac_str ?simple_strtoul(mac_str, &end, 16) : 0;
        if (mac_str) {
            mac_str = (*end) ? end + 1 : end;
        }
    }
    return i;
}


int test(void)
{
     return 0;
}
void set_mtdparts()
{
    char *mtd_str;
    int need_reboot=0;
    if ((mtd_str = getenv("mtdparts")) == NULL)
    {
        need_reboot=1;
    }
    else if(strcmp(mtd_str,MTDPARTS_DEFAULT)!=0)
    {
        need_reboot=1;
    }
    if(need_reboot==1)
    {
        run_command("setenv mtdparts "MTDPARTS_DEFAULT,0);
        run_command("setenv bootcmd "CONFIG_BOOTCOMMAND,0);
        run_command("setenv bootargs "CONFIG_BOOTARGS,0);
        run_command("setenv bootargs 'console=ttyS0,115200 mem=246M'",0);
        run_command("setenv boardmodel 'RTL8382M_8218B_INTPHY_8218B_8214FC_DEMO'",0);
        run_command("setenv ssize_linux 0x00c00000",0);
        run_command("setenv loadaddr  0x81000000",0);
        run_command("setenv bootdelay 3",0);
        run_command("setenv bootcmd 'acc boot'",0);
        run_command("setenv updtuboot 'tftp ${loadaddr} u-boot.bin;sleep 3;sf probe 0;sf erase 0 0x100000;sf write ${fileaddr} 0 ${filesize}'",0);
        run_command("saveenv",0);
        run_command("reset",0);
    }
}

void recover_mfg()
{
#if 0
    int ret;
	init_system();
    if(RC_OK==(ret=HWINFO_read(&hw_info)))
    {
        //hw_info.boot_op
    }
    printf("load mfg");
    sprintf(cmd,"sf read %x %x %x",CONFIG_RUNTIME_LOADADDR,CONFIG_MFG_PART_BASE_ADDR,CONFIG_MFG_PART_SIZE);
    if(0>(ret=run_command(cmd,0)))
    {
        printf("Failed to load recovery partition");
        return;
    }
    //finally, boot linux
    sprintf(cmd,"bootm %x",CONFIG_RUNTIME_LOADADDR);
    run_command(cmd,0);
#endif
}
BOOL_T verify_hwinfo(FS_HW_Info_T *hwinfo)
{
    unsigned short checksum;
    checksum = (unsigned short)CheckSum((unsigned long)hwinfo, sizeof(FS_HW_Info_T) - sizeof(unsigned short));
    if(checksum == 0xffff || checksum != hwinfo->check_sum)
        return FALSE;
    return TRUE;
}
BOOL_T update_uc_by_hwinfo()
{
    if(!verify_hwinfo(&hw_info))
    {
        if(RC_OK!=(HWINFO_read(&hw_info)))
            return FALSE;
    }
    memcpy(sys_info.mac_addr, hw_info.mac_addr, SYS_ADPT_MAC_ADDR_LEN);
    memcpy(sys_info.serial_no,hw_info.serial_no,SYS_ADPT_SERIAL_NO_STR_LEN + 1 );
    memcpy(sys_info.manufacture_date,hw_info.manufacture_date ,SYS_ADPT_MANUFACTURE_DATE_LEN );
    memcpy(sys_info.mainboard_hw_ver,hw_info.agent_hw_ver ,SYS_ADPT_HW_VER_STR_LEN );
    memcpy(sys_info.model_number,hw_info.model_num_by_string ,SYS_ADPT_MODEL_NUMBER_LEN + 1 );
    sys_info.project_id = hw_info.project_id;
    sys_info.board_id = hw_info.board_id;
    sys_info.baudrate = hw_info.baudrate;
    UC_MGR_SetSysInfo(&sys_info);
    return TRUE;
}
void init_system()
{
    int ret;
    char *env_str;

    BOOL_T isColdStart=TRUE;
    unsigned long   magic_number=0;
    set_mtdparts();
    LA_check_uboot_passwd_in_env();
    if(RC_OK!=(ret=HWINFO_read(&hw_info)))
    {
        printf("Warning **Incorrect HWINFO**, reset to default\r\n");
        set_default_hwinfo(&hw_info);
        HWINFO_write(&hw_info);
        update_uc_by_hwinfo();
    }
    //init UC memory
    if(uc_head==NULL)
    {
        //init file mapping file
        fmm = (AMS_FILEMAPPING_MGR_T *)malloc(sizeof(AMS_FILEMAPPING_MGR_T));

        isColdStart=UC_MGR_Init();
        UC_MGR_GetSysInfo(&sys_info);
        if(!isColdStart)
            magic_number=sys_info.magic_number;
        memset(L_CVRT_GET_PTR(uc_head, uc_head->offset[UC_MGR_SYS_INFO_INDEX]), 0, sizeof(UC_MGR_Sys_Info_T));
        memcpy(&sys_info, &default_ucinfo, sizeof(UC_MGR_Sys_Info_T));
        if(isColdStart)
            sys_info.magic_number=SYS_VAL_COLDSTART_MAGIC_NUMBER;
        else if(magic_number!=SYS_VAL_RESTART_MAGIC_NUMBER)
            sys_info.magic_number=SYS_VAL_WARMSTART_MAGIC_NUMBER;

        switch(sys_info.magic_number)
        {
            case SYS_VAL_COLDSTART_MAGIC_NUMBER:
                printf("Cold start\r\n");
                break;
            case SYS_VAL_WARMSTART_MAGIC_NUMBER:
                printf("warm start\r\n");
                break;
            case SYS_VAL_RESTART_MAGIC_NUMBER:
                printf("restart \r\n");
                break;
            default:
                printf("Unknown magic %x\r\n", (unsigned int)sys_info.magic_number);
        }
        //update uboot to uc
        strncpy((char*)sys_info.loader_ver, EXTRA_VERSION, SYS_ADPT_FW_VER_STR_LEN);
        //update hwinfo to uc
        if(TRUE != HWINFO_load())
        {
            printf("Warning! Failed to read hwinfo\r\n");
            if ((env_str = getenv("ethaddr")) == NULL)
            {
                env_str=CONFIG_ETHADDR_STRING;
            }
            scan_mac_from_string(sys_info.mac_addr,env_str);
            UC_MGR_SetSysInfo(&sys_info);
        }
        else
            update_uc_by_hwinfo();

    }
    else
    {
        //printf("simon: UC memory not null, no need to re init\r\n");
        update_uc_by_hwinfo();
    }
    verify_board_id();
    return ;
}
int init_ubifs()
{
    int ret;
    return 0; //no ubifs for SPI  project
    if(!ubi_loaded)
    {
        sprintf(cmd,"ubi part %s",PART_UBIFS_NAME);
        if(0>(ret=run_command(cmd,0)))
            RETURN_ERROR("Failed to load ubi "PART_UBIFS_NAME);

        sprintf(cmd,"ubifsmount %s",VOLUMN_UBIFS_NAME);
        if(0>(ret=run_command(cmd,0)))
            RETURN_ERROR("Failed to mount ubi volumn "VOLUMN_UBIFS_NAME);
        ubi_loaded=TRUE;
    }
    return 0;
}

int acc_boot()
{
    jffs2_boot();
    return 0;
}

int jffs2_load_filemapping()
{
    if(fmm==0)
    {
        RETURN_ERROR("System not init!");
    }
    /*step1: load partition*/
    sprintf(cmd,"chpart nor0,3");
    if(0>run_command(cmd,0))
            RETURN_ERROR("Failed to chpart");

    /*step2: get filemapping file*/
    sprintf(cmd,"ls %s%s",FILE_NAME_PREFIX,FILEMAPPING_FILE_NAME);
    if(0>run_command(cmd,0))
    {
        RETURN_ERROR("Failed to get file mapping file");
    }

    sprintf(cmd,"fsload %x %s%s",(unsigned int)fmm,FILE_NAME_PREFIX,FILEMAPPING_FILE_NAME);
    if(0>run_command(cmd,0))
    {
        RETURN_ERROR("Failed to load file mapping file");
    }
    return 0;
}

int jffs2_boot()
{

    AMS_FILEMAPPING_T *fm;

    int i;
    int size=0;
    char *envalue;
    int filesize=0;
    int ret;

    init_system();
    /*step2: get filemapping file*/
    if(0!=jffs2_load_filemapping())
        return -1;

    envalue=getenv("filesize");
    filesize = simple_strtoul(envalue, NULL, 16);

    /*step3: read boot file and boot*/
    size=0;
    fm=0;
    for(i=0;size<filesize;i++)
    {
        if(i>=MAX_PARTITION_NUMBER)
            break;
        fm=&(fmm->filemapping[i]);
        if(fm->file_type==FS_FILE_TYPE_RUNTIME && fm->startup_flag)
        {
            sprintf(cmd,"ls %s%s",FILE_NAME_PREFIX,GET_STRING(fm->file_name));
            if(0>(ret=run_command(cmd,0)))
            {
               RETURN_ERROR("boot file not found");
            }
            break;
        }
        size+=sizeof(*fm);
    }
    if(!fm || !fm->startup_flag)
	{
        RETURN_ERROR("no startup boot file, use defualt instead??");
    }

    sprintf(cmd,"fsload %x %s%s",CONFIG_RUNTIME_LOADADDR, FILE_NAME_PREFIX,GET_STRING(fm->file_name));
    if(0>(ret=run_command(cmd,0)))
    {
        RETURN_ERROR("Failed to load boot file to memory");
    }
    run_command("setenv acctonruntime 1",0);
    //finally, boot linux
    sprintf(cmd,"bootm %x",CONFIG_RUNTIME_LOADADDR);
    run_command(cmd,0);
	return 0;
}
int ubi_boot()
{
#if 0
    AMS_FILEMAPPING_MGR_T *fmm;
    AMS_FILEMAPPING_T *fm;

    int i;
    int size=0;
    char *envalue;
    int filesize=0;
    int ret;

    init_system();
    init_ubifs();
    /*step2: get filemapping file*/
    sprintf(cmd,"ubifsls %s%s",FILE_NAME_PREFIX,FILEMAPPING_FILE_NAME);
    if(0>(ret=run_command(cmd,0)))
        RETURN_ERROR("Failed to get file mapping file");

    sprintf(cmd,"ubifsload %x %s%s",CONFIG_LOADADDR,FILE_NAME_PREFIX,FILEMAPPING_FILE_NAME);
    if(0>(ret=run_command(cmd,0)))
        RETURN_ERROR("Failed to load file mapping file");
    fmm=(AMS_FILEMAPPING_MGR_T*)CONFIG_LOADADDR;
    envalue=getenv("filesize");
    filesize = simple_strtoul(envalue, NULL, 16);

    /*step3: read boot file and boot*/
    size=0;
    fm=0;
    for(i=0;size<filesize;i++)
    {
        if(i>=MAX_PARTITION_NUMBER)
            break;
        //AMS_PART_get_filemapping(i, fm);
        fm=&(fmm->filemapping[i]);
        if(fm->file_type==FS_FILE_TYPE_RUNTIME && fm->startup_flag)
        {
            sprintf(cmd,"ubifsls %s%s",FILE_NAME_PREFIX,GET_STRING(fm->file_name));
            if(0>(ret=run_command(cmd,0)))
               RETURN_ERROR("boot file not found");
            break;
        }
        size+=sizeof(*fm);
    }
    if(!fm || !fm->startup_flag)
        RETURN_ERROR("no startup boot file, use defualt instead??");


    sprintf(cmd,"ubifsload %x %s%s",CONFIG_RUNTIME_LOADADDR, FILE_NAME_PREFIX,GET_STRING(fm->file_name));
    if(0>(ret=run_command(cmd,0)))
        RETURN_ERROR("Failed to load boot file to memory");
    //finally, boot linux
    sprintf(cmd,"bootm %x",CONFIG_RUNTIME_LOADADDR);
    run_command(cmd,0);
#endif
    return 0;

}
int boot_runtime(char * fname)
{

    AMS_FILEMAPPING_T *fm;
    char *envalue;
    int filesize=0;
    int ret;
    int i;
    int size=0;
    init_system();
    init_ubifs();

    if(fname==0)
    {
        if(0!=jffs2_load_filemapping())
            return -1;
        envalue=getenv("filesize");
        filesize = simple_strtoul(envalue, NULL, 16);

        size=0;
        fm=0;
        printf("list available runtime in system:\r\n");
        for(i=0;size<filesize;i++)
        {
            if(i>=MAX_PARTITION_NUMBER)
                break;
            //AMS_PART_get_filemapping(i, fm);
            fm=&(fmm->filemapping[i]);
            //printf("parsing file %s, type=%d, sf=%d\r\n",GET_STRING(fm->file_name), fm->file_type, fm->startup_flag);
            if(fm->file_type==FS_FILE_TYPE_RUNTIME)
            {
                char star=(fm->startup_flag)?'*':' ';
                printf("%c\t%s\r\n",star,GET_STRING(fm->file_name));

            }

            size+=sizeof(*fm);
        }
    }
    else//if(fname!=0)
    {
        /*step1: load partition*/
        sprintf(cmd,"chpart nor0,3");
        if(0>(ret=run_command(cmd,0)))
                RETURN_ERROR("Failed to chpart");
        sprintf(cmd,"fsload %x %s%s",CONFIG_RUNTIME_LOADADDR, FILE_NAME_PREFIX,fname);
        if(0>(ret=run_command(cmd,0)))
            RETURN_ERROR("Failed to load boot file to memory");

        sprintf(cmd,"bootm %x",CONFIG_RUNTIME_LOADADDR);
        run_command(cmd,0);
    }
    return 0;
}

int acc_recovery()
{
#if 0
    int ret;

    sprintf(cmd,"nand read %x %x %x",CONFIG_RUNTIME_LOADADDR,CONFIG_RECOVERY_PART_BASE_ADDR,CONFIG_RECOVERY_PART_SIZE);
    if(0>(ret=run_command(cmd,0)))
        RETURN_ERROR("Failed to load recovery to memory");
    if(ret!=0)
        return 1;
    //finally, boot linux
    sprintf(cmd,"bootm %x",CONFIG_RUNTIME_LOADADDR);
    run_command(cmd,0);
#endif
    return 0;
}


int
do_bootaccton(cmd_tbl_t * cmdtp, int flag, int argc, char * const argv[])
{
    init_system();
    return do_bootm(cmdtp, flag, argc, argv);
}


int
do_acc(cmd_tbl_t * cmdtp, int flag, int argc, char * const argv[])
{

    int ret=0;
    if (argc > 1)
    {
        switch (argv[1][0])
        {
            case 'b': /*boot*/
                ret=acc_boot();
                break;
            case 'r':/*recovery*/
                recover_mfg();
                break;
            case 'i': /*init system*/
                init_system();
                init_ubifs();
                break;
            case 'f':/*boot specified  runtime*/
                boot_runtime((argc>2)?argv[2]:NULL);
                break;
            case 't':
                test();
                break;
            default:
                return CMD_RET_USAGE;
                break;
        }
    }
    else
        return CMD_RET_USAGE;

    return 0;
}

static const unsigned char hwinfo_var[][ENV_VAR_ITEM_MAX_LEN]  = {
                   "macadr",
                   "serialno",
                   "hwver",
                   "mdate",
                   "servicetag",
                   "modelno",
                   "pid",
                   "bid",
                   "bootop"
};

void verify_board_id()
{
    /*
    MAINBOARD_ECS2000_10T = 0
    MAINBOARD_ECS2000_10PE = 1
    MAINBOARD_ECS2000_10P = 2
    MAINBOARD_ECS2000_28T = 3
    MAINBOARD_ECS2000_28P = 4
    MAINBOARD_ECS2000_28PP = 5
    MAINBOARD_ECS2100-10P-RAI =6
    */
    char* model = getenv("boardmodel");
    int need_reboot=0;
    if(sys_info.board_id ==3 || sys_info.board_id ==4 || sys_info.board_id ==5 )
    {
        if (strcmp("RTL8382M_8218B_INTPHY_8218B_8214FC_DEMO", model) != 0)
        {
            run_command("setenv boardmodel 'RTL8382M_8218B_INTPHY_8218B_8214FC_DEMO",0);
            need_reboot=1;
        }
    }
    else
    {
        if (strcmp("RTL8380M_INTPHY_2FIB_1G_DEMO", model) != 0)
        {
            run_command("setenv boardmodel 'RTL8380M_INTPHY_2FIB_1G_DEMO",0);
            need_reboot=1;
        }
    }
    if(need_reboot)
    {
        run_command("saveenv",0);
        run_command("reset",0);
    }
}

int do_sethwinfo ( cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
    int   i;
    char *name;
    unsigned short checksum;


    name = argv[1];

    FS_HW_Info_T hwinfo;
    if (argc < 3)
        return CMD_RET_USAGE;
    if(uc_head==NULL)
        init_system();
    if(!HWINFO_load())
    {
         return -1;
    }

    hwinfo=hw_info;
    if (strchr(name, '=')) {
        printf ("illegal character '=' in variable name \"%s\"\n", name);
        return RC_ERROR;
    }

    if (strcmp(argv[1], "pid") == 0)
    {
        unsigned long pid;
        BOOL err = FALSE;
        char *t  = argv[2];


        if (strlen(t) > 10 || *t == '-')
            err = TRUE;
        else
        {
            pid = simple_strtoul(t, &t, 0);
            if (*t != ' ' && *t != '\t' && *t != '\0')
                err = TRUE;
            else
            {
                hwinfo.project_id = pid;
                printf("Project ID changed to 0x%08lx\n", pid);
            }
        }

        if (err == TRUE)
        {
            printf("bad Project ID\n");
            return RC_ERROR;
        }
    }

    else if (strcmp(argv[1], "bid") == 0)
    {
        unsigned long bid;
        BOOL err = FALSE;
        char *t  = argv[2];


        if (strlen(t) > 10 || *t == '-')
            err = TRUE;
        else
        {
            bid = simple_strtoul(t, &t, 0);
            if (*t != ' ' && *t != '\t' && *t != '\0')
                err = TRUE;
            else
            {
                hwinfo.board_id = bid;
                printf("Board ID changed to 0x%08lx\n", bid);
            }
        }

        if (err == TRUE)
        {
            printf("bad Board ID\n");
            return RC_ERROR;
        }
    }
    else if (strcmp(argv[1],"hwver") == 0)
    {
        char *t = argv[2];

        if (strlen(t) > SYS_ADPT_HW_VER_STR_LEN)
        {
            printf("bad Hardware Version\n");
            return RC_ERROR;
        }
        strcpy((char *)hwinfo.agent_hw_ver, t);
        printf("Hardware Version changed to %s\n", t);
    }
    else if (strcmp(argv[1],"serialno") == 0)
    {
        char *t = argv[2];

        if (strlen(t) > SYS_ADPT_SERIAL_NO_STR_LEN)
        {
            printf("bad Serial Number\n");
            return RC_ERROR;
        }
        strcpy((char *)hwinfo.serial_no, t);
        printf("Serial Number changed to %s\n", t);
    }
    else if (strcmp(argv[1],"macadr") == 0)
    {
        char *t = argv[2];
        unsigned char m;

        if (ENV_COMMON_check_mac((unsigned char *)t) != RC_OK)
        {
            printf("bad MAC Address\n");
            return RC_ERROR;
        }

        for (i = 0; i < SYS_ADPT_MAC_ADDR_LEN; i++)
        {
            m = (unsigned char)simple_strtoul(t + i * 3, 0, 16);
            hwinfo.mac_addr[i] = m;
        }

        printf("MAC Address changed to %02X:%02X:%02X:%02X:%02X:%02X\n",
               hwinfo.mac_addr[0], hwinfo.mac_addr[1],
               hwinfo.mac_addr[2], hwinfo.mac_addr[3],
               hwinfo.mac_addr[4], hwinfo.mac_addr[5]);
    }
    else if (strcmp(argv[1], "modelno") == 0)
    {
        char *t = argv[2];

        if (strlen(t) > SYS_ADPT_MODEL_NUMBER_LEN)
        {
            printf("bad Model Number\n");
            return RC_ERROR;
        }
        strcpy((char *)hwinfo.model_num_by_string, t);
        printf("Model Number changed to %s\n", t);
    }
    else if (strcmp(argv[1],"servicetag") == 0)
    {
        char *t = argv[2];

        if (strlen(t) > SYS_ADPT_SERIAL_NO_STR_LEN)
        {
            printf("bad Service Tag\n");
            return RC_ERROR;
        }
        strcpy((char *)hwinfo.service_tag, t);
        printf("Service Tag changed to %s\n", t);
    }
    else if (strcmp(argv[1], "mdate") == 0)
    {
        char *t  = argv[2];
        int  len = strlen(t);
        BOOL err = FALSE;
        unsigned long v;

        if (len > SYS_ADPT_MANUFACTURE_DATE_LEN || len < 8)
            err = TRUE;
        else
        {
            char *t2 = t;
            v = simple_strtoul(t2, &t2, 10);

            if (*t2 == '-' && v >= 1970 && v <= 2099)
            {
                v = simple_strtoul(t2 + 1, &t2, 10);

                if (*t2 == '-' && v >= 1 && v <= 12)
                {
                    v = simple_strtoul(t2 + 1, &t2, 10);

                    if ((*t2 == ' ' || *t2 == '\0' || *t2 == '\t') && v >= 1 && v <= 31)
                    {
                        strcpy((char *)hwinfo.manufacture_date, t);
                        printf("Manufacture Date changed to %s\n", t);
                        goto end;
                    }
                }
            }
            err = TRUE;
        }
end:
        if (err == TRUE)
        {
            printf("bad Manufacture Date\n");
            return RC_ERROR;
        }
    }
    else if(strcmp(argv[1], "bootop") == 0)
    {
        char *t  = argv[2];

        hwinfo.special_boot_op = simple_strtoul(t, NULL, 16);
    }
    else if (strcmp(argv[1], "capability") == 0)
    {
        unsigned int capability;
        BOOL err = FALSE;
        char *t  = argv[2];


        if (strlen(t) > 1 || *t == '-')
            err = TRUE;
        else
        {
            capability = simple_strtoul(t, &t, 0);
            if (*t != ' ' && *t != '\t' && *t != '\0')
                err = TRUE;
            else
            {
                hwinfo.capability = capability;
                printf("capability changed to 0x%01x\n", capability);
            }
        }

        if (err == TRUE)
        {
            printf("bad capability\n");
            return RC_ERROR;
        }
    }
    else
    {
        printf("Bad argument: %s\n", argv[1]);
        return CMD_RET_USAGE;
    }

    /* Update checksum */
    checksum = CheckSum((unsigned long)&hwinfo, sizeof(FS_HW_Info_T) - sizeof(unsigned short));
    hwinfo.check_sum = checksum;

    /*write to hwinfo partition*/
    if(RC_OK!=HWINFO_write(&hwinfo))
        return -1;
    /*update hw_info variable*/
    if(TRUE == HWINFO_load())
        update_uc_by_hwinfo();

    return RC_OK;
}

int do_printhwinfo( cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
    int i, k, len;
    int rcode = 0;


    if(!HWINFO_load())
    {
        printf("hardware information table is incorrect!\n");
        goto END;
    }



    if (argc == 1)
    {
        /* Print all hwinfo variables   */
        printf("----------------------------------------\n");
        printf("%-21.21s=0x%08lx\n", "Project ID",hw_info.project_id);
        printf("%-21.21s=0x%08lx\n", "Board ID",hw_info.board_id);
        printf("%-21.21s=%s\n", "Hardware Version",hw_info.agent_hw_ver);
        printf("%-21.21s=%s\n", "Serial Number",hw_info.serial_no);
        printf("%-21.21s=%02X:%02X:%02X:%02X:%02X:%02X\n", "Mac Address",
            hw_info.mac_addr[0],
            hw_info.mac_addr[1],
            hw_info.mac_addr[2],
            hw_info.mac_addr[3],
            hw_info.mac_addr[4],
            hw_info.mac_addr[5]);
        printf("%-21.21s=%s\n", "Manufacture Date",GET_STRING(hw_info.manufacture_date));
        printf("%-21.21s=%s\n", "Model Number",hw_info.model_num_by_string);
        printf("%-21.21s=%s\n", "Service Tag",hw_info.service_tag);
        printf("%-21.21s=0x%08lx\n", "Boot OP", hw_info.special_boot_op);
        printf("%-21.21s=%01x\n", "Capability",hw_info.capability);

        printf("----------------------------------------\n");
        goto END;
    }

    /*print single variable*/
    for (i=1; i<argc; ++i)
    {
        char *name = argv[i];

        k = -1;
        for (i=0; i < sizeof(hwinfo_var)/ENV_VAR_ITEM_MAX_LEN; i++)
        {
            len = strlen(name);

            if(memcmp(hwinfo_var[i], name, len) == 0)
            {
                switch(i)
                {
                case 0:
                    printf("%-21.21s=%02X:%02X:%02X:%02X:%02X:%02X\n", name,
                        hw_info.mac_addr[0],
                        hw_info.mac_addr[1],
                        hw_info.mac_addr[2],
                        hw_info.mac_addr[3],
                        hw_info.mac_addr[4],
                        hw_info.mac_addr[5]);
                    k++;
                    break;
                case 1:
                    printf("%-21.21s=%s\n", name, hw_info.serial_no);
                    k++;
                    break;
                case 2:
                    printf("%-21.21s=%s\n", name, hw_info.agent_hw_ver);
                    k++;
                    break;
                case 3:
                    printf("%-21.21s=%s\n", name, hw_info.manufacture_date);
                    k++;
                    break;
                case 4:
                    printf("%-21.21s=%s\n", name, hw_info.service_tag);
                    k++;
                    break;
                case 5:
                    printf("%-21.21s=%s\n", name, hw_info.model_num_by_string);
                    k++;
                    break;
                case 6:
                    printf("%-21.21s=%ld\n", name, hw_info.project_id);
                    k++;
                    break;
                case 7:
                    printf("%-21.21s=%ld\n", name, hw_info.board_id);
                    k++;
                    break;
                case 8:
                    printf("%-21.21s=%ld\n", name, hw_info.special_boot_op);
                    k++;
                    break;
                default:
                    break;
                }
            }
        }
        if (k < 0) {
            printf("\"%s\" not defined\n", name);
            rcode ++;
        }
    }

END:
    return rcode;
}

#define UI_TIMEOUT_STEP 1
#define UI_TIMEOUT_BASE 1000    /* 1 ms */
#define UI_WAIT_TIME    2000   /* 2 seconds */
static int LA_check_uboot_passwd_in_env(void)
{
#ifdef CONFIG_USR_PRIVILEGED_MODE
    char* passwd_p;

    if (gd->env_valid == 0)
    {
        /* uboot environment in the flash is invalid, need to write a valid
         * one.
         */
        setenv ("passwd", LA_LOADER_BACKDOOR_PASSWORD_ENCRYPT);
        saveenv();
    }
    else
    {
        /* uboot environment in the flash is valid, check whether "passwd" is valid
         */
        passwd_p = getenv("passwd");
        if (passwd_p == NULL)
        {
            setenv ("passwd", LA_LOADER_BACKDOOR_PASSWORD_ENCRYPT);
            saveenv();
        }
    }
#endif
    return 0;
}

static void Encrypt(char * data)
{
    char charset[] = "abcdefghijklmnopqrstuvwxyz1234567890";
    int i, index, charsetLen, dataLen;

    charsetLen = strlen(charset);
    dataLen = strlen(data);

    for(i=0; i<dataLen; i++)
    {
       index = data[i]%(char)charsetLen;
       data[i] = charset[index];
    }
}

void Accton_Pre_loadermain(void)
{
    char passwd[LA_MAX_PASSWORD_LENGTH+1];
    if(NULL!=getenv("acctonruntime"))
    {
        init_system();
        puts("Password : ");
        memset(passwd, 0x0, LA_MAX_PASSWORD_LENGTH+1);
        LA_get_password(passwd, UI_WAIT_TIME*5, LA_MAX_PASSWORD_LENGTH);
        putc('\n');
        Encrypt(passwd);
        if(getenv("passwd")!=NULL)
        {
#ifdef CONFIG_USR_PRIVILEGED_MODE
            if(0==strcmp(passwd, getenv("passwd")))
#else
            if(0==strcmp(passwd, LA_LOADER_BACKDOOR_PASSWORD_ENCRYPT))
#endif
            {
                return;
            }
        }
        run_command("reset",0);
    }
}
#ifdef CONFIG_USR_PRIVILEGED_MODE
static int LA_set_privi_mode(unsigned long mode)
{
    LA_privileged_mode=mode;
    return 0;
}

static unsigned long LA_get_privi_mode(void)
{
    if(NULL!=getenv("acctonruntime"))
    return LA_privileged_mode;
    else
        return TRUE;
}
int do_enable_privi_mode ( cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
    char passwd[LA_MAX_PASSWORD_LENGTH+1];

    if(argc < 1)
        return -1;

    puts("Password : ");
    memset(passwd, 0x0, LA_MAX_PASSWORD_LENGTH+1);
    LA_get_password(passwd, UI_WAIT_TIME*5, LA_MAX_PASSWORD_LENGTH);
    putc('\n');

    /* check password */
    Encrypt(passwd);
    if(strcmp(passwd, LA_LOADER_PRIVILEGED_MODE_PASSWORD_ENCRYPT)){
        puts("Incorrect Password.\n");
        return -1;
    }
    puts("Privileged mode is enabled.\n");
    return LA_set_privi_mode(TRUE);
}
U_BOOT_CMD(
        enable, CFG_MAXARGS, 1,do_enable_privi_mode,
        "enable\t - enable privileged mode.\n",
        "\n"
);
int do_passwd ( cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
    //int i, k, len;

    char passwd[LA_MAX_PASSWORD_LENGTH+1];
    char passwd_retype[LA_MAX_PASSWORD_LENGTH+1];

    if(argc < 1)
        return -1;

    if(!hw_info.capability)
    {
        printf("Operation is not allowed. \n");
        return -1;
    }


    printf("New password:");
    memset(passwd, 0x0, LA_MAX_PASSWORD_LENGTH+1);
    LA_get_password(passwd, UI_WAIT_TIME*5, LA_MAX_PASSWORD_LENGTH);
    putc('\n');

    if(strlen(passwd)<6)
    {
        printf("Illegal password. \n");
        return -1;
    }

    printf("Retype New password:");
    memset(passwd_retype, 0x0, LA_MAX_PASSWORD_LENGTH+1);
    LA_get_password(passwd_retype, UI_WAIT_TIME*5, LA_MAX_PASSWORD_LENGTH);
    putc('\n');

    /* check password same */
    if(strcmp(passwd, passwd_retype)){
        puts("Password does not match.\n");
        return -1;
    }

    /*save passwd to  evirnment*/
    Encrypt(passwd);
    setenv ("passwd", passwd);
    saveenv();
    puts("Password is changed successfully.\n");

    return 0;
}

U_BOOT_CMD(
     passwd, CFG_MAXARGS, 1,do_passwd,
     "passwd\t - change loader password.\n",
     "\n"
);

#endif

BOOL_T Accton_Chk_PrivilegedDeny(char *cmd)
{
#ifdef CONFIG_USR_PRIVILEGED_MODE
    int i;
    char pcmd[][CONFIG_SYS_CBSIZE]={
            "passwd",
            "mw",
            "nm",
            "sethwinfo",
            "saveenv",
            "run",
            "format",
            "sf write",
            "sf erase",
            "sf update"
            "nand"

    };
    if(TRUE == LA_get_privi_mode())
        return FALSE;
    for(i=0;i<sizeof(pcmd)/CONFIG_SYS_CBSIZE;i++)
    {
        if(memcmp(cmd,pcmd[i],strlen(pcmd[i]))==0)
        {
            printf("Permission denied.\n");
            return TRUE;
        }
    }
#endif
    return FALSE;
}

static int LA_get_password(char *passwd, uint timeout, uint max_size)
{
    uint    length;
    int     endline;
    char    ch;

    endline = FALSE;
    length = 0;

    if(!passwd || (max_size == 0))
        return 0;

    while ((timeout > 0) && (endline != TRUE)){
        if(timeout < UI_TIMEOUT_STEP)
            timeout = UI_TIMEOUT_STEP;

        timeout -= UI_TIMEOUT_STEP;
        udelay(UI_TIMEOUT_BASE);

        if(!tstc()) /* wait for user if there is no incoming data */
            continue;


        ch = getc();
        switch (ch){
            case '\n':
            case '\r':
            case '\0':
                passwd[length] = '\0';
                endline = TRUE;
                break;

            case '\b':
                if(length>0){
                    /* erase the last input byte */
                    length -= 1;
                    printf("\b \b%c",'\0');
                }
                break;

            case 0x03:    /* ESCAPE */
                length = 0;
                passwd[length] = '\0';
                endline = TRUE;
                break;

            default:
                if(length < max_size){
                    passwd[length++] = ch;
                    putc('*');
                }
        }
    }

    if(timeout == 0)
        length = 0;

    return length;
}




U_BOOT_CMD(acc, CONFIG_SYS_MAXARGS, 1, do_acc, "accton custom command",
    "acc boot: boot accton runtime\n"
    "acc recovery: boot default runtime\n"
    "acc file: list/boot runtime file\n"
    "\n"
    "\n");



U_BOOT_CMD(bootaccton, CONFIG_SYS_MAXARGS, 1, do_bootaccton, "accton bootm command wrapper",
    "same usage as bootm, but init accton data structure first\n"
    "use bootm /? to find out how to use\n"
    "\n");

U_BOOT_CMD(
    printhwinfo, CONFIG_SYS_MAXARGS, 1, do_printhwinfo,
    "printhwinfo  - print hardware information variables\n",
    "\n    - print values of all hardware information variables\n"
    "print_hwinfo name ...\n"
    "    - print value of hardware information variable 'name'\n"
    "\n"
);





U_BOOT_CMD(
    sethwinfo, CONFIG_SYS_MAXARGS, 0,   do_sethwinfo,
    "sethwinfo    - set hardware information variables\n",
    "name value ...\n"
    "    - set hardware information variable 'name' to 'value ...'\n"
    "Following hardware information can be set:\n"
    "  pid: Project ID\n"
    "  bid: Board ID\n"
    "  hwver: Hardware Version\n"
    "  serialno: Serial Number\n"
    "  macadr: MAC Address\n"
    "  modelno: Model Number\n"
    "  servicetag: Service Tag\n"
    "  mdate: Manufacture Date\n"
    "  bootop: Set special boot op\n"
    "  capability: set capability flag\n"
);




